/*
 * Cpuautoplug driver for the Loongson-3 processors
 *
 * Copyright (C) 2006 - 2011 Lemote Inc.
 * Author: Huacai Chen, chenhc@lemote.com
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 */
#include <linux/module.h>
#include <linux/cpu.h>
#include <linux/sched.h>
#include <linux/tick.h>
#include <linux/kernel_stat.h>
#include <linux/platform_device.h>

#include <asm/clock.h>

#include <loongson.h>

/*
 * CPU Autoplug enabled ?
 */
int autoplug_enabled = 1;
int autoplug_adjusting = 0;

struct cpu_autoplug_info {
	cputime64_t prev_idle;
	cputime64_t prev_wall;
	struct delayed_work work;
	unsigned int sampling_rate;
	int maxcpus;   /* max cpus for autoplug */
	int mincpus;   /* min cpus for autoplug */
	int dec_reqs;  /* continous core-decreasing requests */
};

struct cpu_autoplug_info ap_info;

static ssize_t show_autoplug_enabled(struct sysdev_class *class,
				     struct sysdev_class_attribute *attr,
				     char *buf)
{
	return sprintf(buf, "%d\n", autoplug_enabled);
}

static ssize_t store_autoplug_enabled(struct sysdev_class *class,
				      struct sysdev_class_attribute *attr,
				      const char *buf, size_t count)
{
	char val[5];
	int n;

	memcpy(val, buf, count);
	n = simple_strtol(val, NULL, 0);

	if(n > 1 || n < 0)
		return -EINVAL;

	autoplug_enabled = n;

	return count;
}

static ssize_t show_autoplug_maxcpus(struct sysdev_class *class,
				     struct sysdev_class_attribute *attr,
				     char *buf)
{
	return sprintf(buf, "%d\n", ap_info.maxcpus);
}

static ssize_t store_autoplug_maxcpus(struct sysdev_class *class,
				      struct sysdev_class_attribute *attr,
				      const char *buf, size_t count)
{
	char val[5];
	int n;

	memcpy(val, buf, count);
	n = simple_strtol(val, NULL, 0);

	if(n > NR_CPUS || n < ap_info.mincpus)
		return -EINVAL;

	ap_info.maxcpus = n;

	return count;
}

static ssize_t show_autoplug_mincpus(struct sysdev_class *class,
				     struct sysdev_class_attribute *attr,
				     char *buf)
{
	return sprintf(buf, "%d\n", ap_info.mincpus);
}

static ssize_t store_autoplug_mincpus(struct sysdev_class *class,
				      struct sysdev_class_attribute *attr,
				      const char *buf, size_t count)
{
	char val[5];
	int n;

	memcpy(val, buf, count);
	n = simple_strtol(val, NULL, 0);

	if(n > ap_info.maxcpus || n < 1)
		return -EINVAL;

	ap_info.mincpus = n;

	return count;
}

static ssize_t show_autoplug_sampling_rate(struct sysdev_class *class,
				     struct sysdev_class_attribute *attr,
				     char *buf)
{
	return sprintf(buf, "%d\n", ap_info.sampling_rate);
}

#define SAMPLING_RATE_MAX 1000
#define SAMPLING_RATE_MIN 600

static ssize_t store_autoplug_sampling_rate(struct sysdev_class *class,
				      struct sysdev_class_attribute *attr,
				      const char *buf, size_t count)
{
	char val[6];
	int n;

	memcpy(val, buf, count);
	n = simple_strtol(val, NULL, 0);

	if(n > SAMPLING_RATE_MAX || n < SAMPLING_RATE_MIN)
		return -EINVAL;

	ap_info.sampling_rate = n;

	return count;
}

static ssize_t show_autoplug_available_values(struct sysdev_class *class,
				     struct sysdev_class_attribute *attr,
				     char *buf)
{
	return sprintf(buf, "enabled: 0-1\nmaxcpus: 1-%d\nmincpus: 1-%d\nsampling_rate: %d-%d\n",
			NR_CPUS, NR_CPUS, SAMPLING_RATE_MIN, SAMPLING_RATE_MAX);
}

static SYSDEV_CLASS_ATTR(enabled, 0644, show_autoplug_enabled, store_autoplug_enabled);
static SYSDEV_CLASS_ATTR(maxcpus, 0644, show_autoplug_maxcpus, store_autoplug_maxcpus);
static SYSDEV_CLASS_ATTR(mincpus, 0644, show_autoplug_mincpus, store_autoplug_mincpus);
static SYSDEV_CLASS_ATTR(sampling_rate, 0644, show_autoplug_sampling_rate, store_autoplug_sampling_rate);
static SYSDEV_CLASS_ATTR(available_values, 0444, show_autoplug_available_values, NULL);

static struct attribute *cpuclass_default_attrs[] = {
	&attr_enabled.attr,
	&attr_maxcpus.attr,
	&attr_mincpus.attr,
	&attr_sampling_rate.attr,
	&attr_available_values.attr,
	NULL
};

static struct attribute_group cpuclass_attr_group = {
	.attrs = cpuclass_default_attrs,
	.name = "cpuautoplug",
};

#ifndef MODULE
/*
 * Enable / Disable CPU Autoplug
 */
static int __init setup_autoplug(char *str)
{
	if (!strcmp(str, "off"))
		autoplug_enabled = 0;
	else if (!strcmp(str, "on"))
		autoplug_enabled = 1;
	else
		return 0;
	return 1;
}

__setup("autoplug=", setup_autoplug);

#endif

static struct workqueue_struct *kautoplugd_wq;

static inline cputime64_t get_idle_time_jiffy(cputime64_t *wall)
{
	unsigned int cpu;
	cputime64_t idle_time = cputime64_zero;
	cputime64_t cur_wall_time;
	cputime64_t busy_time;

	cur_wall_time = jiffies64_to_cputime64(get_jiffies_64());

	for_each_online_cpu(cpu) {
		busy_time = cputime64_add(kstat_cpu(cpu).cpustat.user,
			kstat_cpu(cpu).cpustat.system);

		busy_time = cputime64_add(busy_time, kstat_cpu(cpu).cpustat.irq);
		busy_time = cputime64_add(busy_time, kstat_cpu(cpu).cpustat.softirq);
		busy_time = cputime64_add(busy_time, kstat_cpu(cpu).cpustat.steal);
		busy_time = cputime64_add(busy_time, kstat_cpu(cpu).cpustat.nice);

		idle_time = cputime64_add(idle_time, cputime64_sub(cur_wall_time, busy_time));
	}

	if (wall)
		*wall = (cputime64_t)jiffies_to_usecs(cur_wall_time);

	return (cputime64_t)jiffies_to_usecs(idle_time);
}

static inline cputime64_t get_idle_time(cputime64_t *wall)
{
	unsigned int cpu;
	u64 idle_time = 0;

	for_each_online_cpu(cpu) {
		idle_time += get_cpu_idle_time_us(cpu, wall);
		if (idle_time == -1ULL)
			return get_idle_time_jiffy(wall);
	}

	return idle_time;
}

static void increase_cores(int cur_cpus)
{
	int target_cpu;

	if(cur_cpus == ap_info.maxcpus)
		return;

	target_cpu = cpumask_next_zero(0, cpu_online_mask);
	cpu_hotplug_driver_lock();
	cpu_up(target_cpu);
	cpu_hotplug_driver_unlock();
}


static void decrease_cores(int cur_cpus)
{
	int target_cpu;

	if(cur_cpus == ap_info.mincpus)
		return;

	target_cpu = find_last_bit(cpumask_bits(cpu_online_mask), NR_CPUS);
	cpu_hotplug_driver_lock();
	cpu_down(target_cpu);
	cpu_hotplug_driver_unlock();
}

#define INC_THRESHOLD 95
#define DEC_THRESHOLD 10

static void do_autoplug_timer(struct work_struct *work)
{
	cputime64_t cur_wall_time = 0, cur_idle_time;
	unsigned int idle_time, wall_time;
	int delay, load, nr_cpus = num_online_cpus();

	BUG_ON(smp_processor_id() != 0);
	delay = msecs_to_jiffies(ap_info.sampling_rate);
	if(!autoplug_enabled || system_state != SYSTEM_RUNNING)
		goto out;

	autoplug_adjusting = 1;

	/* user limits */
	if(nr_cpus > ap_info.maxcpus) {
		decrease_cores(nr_cpus);
		autoplug_adjusting = 0;
		goto out;
	}
	if(nr_cpus < ap_info.mincpus) {
		increase_cores(nr_cpus);
		autoplug_adjusting = 0;
		goto out;
	}

	/* based on cpu load */
	cur_idle_time = get_idle_time(&cur_wall_time);
	if (cur_wall_time == 0)
		cur_wall_time = jiffies64_to_cputime64(get_jiffies_64());

	wall_time = (unsigned int) cputime64_sub(cur_wall_time, ap_info.prev_wall);
	ap_info.prev_wall = cur_wall_time;

	idle_time = (unsigned int) cputime64_sub(cur_idle_time, ap_info.prev_idle);
	idle_time += wall_time * (NR_CPUS - nr_cpus);
	ap_info.prev_idle = cur_idle_time;

	if (unlikely(!wall_time || wall_time * NR_CPUS < idle_time)) {
		autoplug_adjusting = 0;
		goto out;
	}

	load = 100 * (wall_time * NR_CPUS - idle_time) / wall_time;

	if(load < (nr_cpus - 1) * 100 - DEC_THRESHOLD) {
		if(ap_info.dec_reqs <= 2)
			ap_info.dec_reqs++;
		else {
			ap_info.dec_reqs = 0;
			decrease_cores(nr_cpus);
		}
	}
	else {
		ap_info.dec_reqs = 0;
		if(load > (nr_cpus - 1) * 100 + INC_THRESHOLD)
			increase_cores(nr_cpus);
	}

	autoplug_adjusting = 0;
out:
	queue_delayed_work_on(0, kautoplugd_wq, &ap_info.work, delay);
}

static struct platform_device_id platform_device_ids[] = {
	{
		.name = "ls3_cpuautoplug",
	},
	{}
};

MODULE_DEVICE_TABLE(platform, platform_device_ids);

static struct platform_driver platform_driver = {
	.driver = {
		.name = "ls3_cpuautoplug",
		.owner = THIS_MODULE,
	},
	.id_table = platform_device_ids,
};

static int __init cpuautoplug_init(void)
{
	int ret, delay;

	ret = sysfs_create_group(&cpu_sysdev_class.kset.kobj, &cpuclass_attr_group);
	if (ret)
		return ret;

	/* Register platform stuff */
	ret = platform_driver_register(&platform_driver);
	if (ret)
		return ret;

	pr_info("cpuautoplug: Loongson-3A CPU autoplug driver.\n");

	ap_info.maxcpus = setup_max_cpus;
	ap_info.mincpus = 1;
	ap_info.dec_reqs = 0; 
	ap_info.sampling_rate = 720;  /* 720 ms */
#ifndef MODULE
	delay = msecs_to_jiffies(ap_info.sampling_rate * 24);
#else
	delay = msecs_to_jiffies(ap_info.sampling_rate * 8);
#endif
	kautoplugd_wq = alloc_workqueue("kautoplug", WQ_NON_REENTRANT, 1);
	if (!kautoplugd_wq) {
		printk(KERN_ERR "Creation of kautoplugd failed\n");
		return -EFAULT;
	}
	INIT_DELAYED_WORK_DEFERRABLE(&ap_info.work, do_autoplug_timer);
	queue_delayed_work_on(0, kautoplugd_wq, &ap_info.work, delay);

	return ret;
}

static void __exit cpuautoplug_exit(void)
{
	cancel_delayed_work_sync(&ap_info.work);
	destroy_workqueue(kautoplugd_wq);
	platform_driver_unregister(&platform_driver);
	sysfs_remove_group(&cpu_sysdev_class.kset.kobj, &cpuclass_attr_group);
}

late_initcall(cpuautoplug_init);
module_exit(cpuautoplug_exit);

MODULE_AUTHOR("Huacai Chen <chenhc@lemote.com>");
MODULE_DESCRIPTION("cpuautoplug driver for Loongson3A");
MODULE_LICENSE("GPL");
