/*
 * Copyright (C) 2009 Lemote Inc.
 * Author: Wu Zhangjin, wuzhangjin@gmail.com
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/err.h>
#include <linux/platform_device.h>
#include <loongson_hwmon.h>
#include <asm/bootinfo.h>

/* Loongson 3A/2GQ series laptop use wpce775l as Embeded Controller */
static struct platform_device wpce775l_chip = {
	.name = "wpce775l",
	.id = -1,
};

/* temprature sensor EMC1412 used in A1004, A1101, A1205 */
static struct platform_device emc1412_sensor = {
	.name = "EMC1412",
	.id = 0,
};

static struct platform_device a1004_fan1 = {
	.name = "a1004-fan1",
	.id = 0,
};

/* sb700-fan1 in A1101, A1205 */
static struct platform_device sb700_fan1 = {
	.name = "sb700-fan1",
	.id = 0,
};

/* sb700-fan2 in A1101 */
static struct platform_device sb700_fan2 = {
	.name = "sb700-fan2",
	.id = 0,
};

/*
 * A1004 Fan1 controler polic
 *
 * Fan is controlled by EC in A1004 pruducts, but EC can not get the current 
 * cpu temprature which used for adjusting the current fan speed.
 *
 * So, kernel read the CPU temprature and notify it to EC per second, 
 * that's all!
 */
struct loongson_fan_policy a1004_fan1_default_policy = {
	.type = KERNEL_HELPER_POLICY,
	.adjust_period = 1,
	.depend_temp = loongson3_cpu_temp,
};

/* 
 * A1101 Fan1/Fan2 control policy 
 * 
 * Fan1 & Fan2 keep at half-speed
 *
 */
struct loongson_fan_policy a1101_fan1_default_policy = {
	.type = CONSTANT_SPEED_POLICY,
	.percent = 50,
};

struct loongson_fan_policy a1101_fan2_default_policy = {
	.type = CONSTANT_SPEED_POLICY,
	.percent = 50,
};

/*
 * A1205 Fan1 at step mode
 *
 * up_step array    |   down_step array
 *                  |    
 * [min, 60), 50%   |   (min, 57), 50%
 * [60, 65),  60%   |   [57 ,62),  60%
 * [65, 70),  70%   |   [62 ,70),  70%
 * [70, 80),  80%   |   [70 ,75),  80%
 * [80, max), 100%  |   [75,max),  100%
 *
 */
struct loongson_fan_policy a1205_fan1_default_policy = {
	.type = STEP_SPEED_POLICY,
	.adjust_period = 1,
	.depend_temp = loongson3_cpu_temp,
	.up_step_num = 5,
	.down_step_num = 5,
	.up_step = {
			{MIN_TEMP,    60,   50},
			{   60,       65,   60},
			{   65,       70,   70},
			{   70,       80,   80},
			{   80,    MAX_TEMP,100},
		   },
	.down_step = {
			{MIN_TEMP, 57,    50},
			{   57,    62,    60},
			{   62,    70,    70},
			{   70,    75,    80},
			{   75, MAX_TEMP, 100},
		     },
};

static int __init loongson3a_platform_init(void)
{
	/* temprature info */
	switch (mips_machtype) {
	case MACH_LEMOTE_3A_A1004:
	case MACH_LEMOTE_3A_A1217:
		/* thermal sensor register and interface init */
		loongson_temp_info.get_cpu_temp = loongson3_cpu_temp; 
		loongson_temp_info.get_nb_temp = emc1412_external_temp;
		loongson_temp_info.get_mb_temp = emc1412_internal_temp;
		loongson_temp_info.get_sb_temp = NULL; // not implement now
		platform_device_register(&emc1412_sensor);

		/* fan sensor register and choose fan control policy */
		platform_device_register(&a1004_fan1);
		loongson_fan1_ops.fan_policy = &a1004_fan1_default_policy;

		/* for lempte-laptop */
		platform_device_register(&wpce775l_chip);
		break;
	case MACH_LEMOTE_3A_A1101:
		/* thermal sensor register and interface init */
		loongson_temp_info.get_cpu_temp = loongson3_cpu_temp; 
		loongson_temp_info.get_nb_temp = emc1412_external_temp;
		loongson_temp_info.get_mb_temp = emc1412_internal_temp;
		loongson_temp_info.get_sb_temp = NULL; // not implement now
		platform_device_register(&emc1412_sensor);

		/* fan sensor register and choose fan control policy */
		platform_device_register(&sb700_fan1);
		loongson_fan1_ops.fan_policy = &a1101_fan1_default_policy;
		platform_device_register(&sb700_fan2);
		loongson_fan2_ops.fan_policy = &a1101_fan2_default_policy;
		break;
	case MACH_LEMOTE_2GQ_A1201:
		/* thermal sensor register and interface init */
		loongson_temp_info.get_cpu_temp = loongson3_cpu_temp; 
		loongson_temp_info.get_nb_temp = emc1412_external_temp;
		loongson_temp_info.get_mb_temp = emc1412_internal_temp;
		loongson_temp_info.get_sb_temp = NULL; // not implement now
		platform_device_register(&emc1412_sensor);

		/* A1201 without fan */

		/* for lempte-laptop */
		platform_device_register(&wpce775l_chip);
		break;
	case MACH_LEMOTE_2GQ_A1205:
		/* thermal sensor register and interface init */
		loongson_temp_info.get_cpu_temp = loongson3_cpu_temp; 
		loongson_temp_info.get_nb_temp = emc1412_external_temp;
		loongson_temp_info.get_mb_temp = emc1412_internal_temp;
		loongson_temp_info.get_sb_temp = NULL; // not implement now
		platform_device_register(&emc1412_sensor);

		/* fan sensor register and choose fan control policy */
		platform_device_register(&sb700_fan1);
		loongson_fan1_ops.fan_policy = &a1205_fan1_default_policy;
		break;
	default:
		break;
	}

	return 0;
}

arch_initcall(loongson3a_platform_init);
